<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-as-a-service
* @subpackage models
* @filesource
*/

/** */
require_once 'folder.php';
require_once 'message.php';
require_once 'user.php';

/**
* @author M. Gibbs <gibbs_margaret@bah.com>
* @package direct-as-a-service
* @subpackage models
*/
class Mailbox extends Entity {
	static $database_group = 'mail_db';
	static $table = 'mailboxes';	
	
	/** 
	* The configuration for this model's relationships.
	*
	* Entities make use of PHP's magical methods ({@link __get}, {@link __ set}, and {@__call} to create virtual variables and methods that will search for
	* related entities, count the related entities, and provide a boolean true/false as to whether there are any entities related to this entity via this
	* relationship.  For example, for the inbox_message relationship, we have all of these variables and methods available just by defining the relationship:
	*
	* <code>
	* $mailbox = Mailbox::find_one( array('username' => jswift') );
	* $mailbox->inbox_messages; //an array of all the messages in this mailbox's inbox 
	* $mailbox->inbox_messages( array( 'subject' => 'A Modest Proposal' )); //an array of all the messages in this mailbox's inbox that match the search criteria
	* $mailbox->inbox_message_count; //the number of messages in the inbox
	* $mailbox->inbox_message_count( array( 'subject' => 'A Modest Proposal' )); //number of messages that match the search criteria
	* $mailbox->has_inbox_messages; //true if there are any inbox messages
	* $mailbox->has_inbox_messages( array( 'subject' => 'A Modest Proposal' )); //true if there are messages that match the search criteria
	* </code>
	*
	* Note that the Mailbox class additionally allows the developer to append a message part (see {@link Message}) to the end of a message relationship to 
	* limit the selection just the fields relevant to the part.
	*
	* <code>
	* $mailbox->inbox_message_headers; //only the header fields will be selected for the inbox messages
	* $mailbox->inbox_message_headers( array('timestamp < ' => mktime() ) ); //only the header fields will be selected for the search results
	* </code>
	*
	* This default behavior will be overriden if you define any methods/variables by these names for this class.  Also note that since these are not *actually* 
	* variables on the class, isset() will always return false.  To see the details of how the values for these variables/methods are determined, see 
	* {@link __get} and {@link __call}.
	*
	* For more information on how to set up or configure relationships, see {@link relationships()}. 
	*
	* @var array
	*/
	protected static $_relationships = array( 'folder' => array('type' => 'has_many'),
											  'message' => array('type' => 'has_many'),
											  'inbox_message' => array( 'type' => 'has_many', 'model' => 'message',  'condition' => 'folder_id IS NULL AND sent=0 AND draft=0 AND archived=0'),
											  'archived_message' => array( 'type' => 'has_many', 'model' => 'message', 'condition' => 'archived=1'),
											  'sent_message' => array( 'type' => 'has_many', 'model' => 'message', 'condition' => 'sent=1 AND archived=0'),
											  'draft' => array( 'type' => 'has_many', 'model' => 'message', 'condition' => 'draft=1 AND archived=0'),
											  'user' => array( 'type' => 'belongs_to', 'related_foreign_key' => 'name', 'key_for_relationship' => 'username' ),
											 );
											 
	protected $_property_validation_rules = array( 'name' => 'nonempty_string',
												   'is_group' => 'boolean',
												   'is_active' => 'boolean',
												   'facility_id' => 'null|nonzero_unsigned_integer' );												 
	
	
//////////////////////////////////////////////////////////////////////////////////////////
// INSTANCE METHODS 
// These methods assume that the instance represents a mailbox record in the database.
//////////////////////////////////////////////////////////////////////////////////////////			
	
	function add_folder($values){
		if(!$this->is_active) return $this->error->warning("I can't add a new folder to ".$this->describe().' while it is disabled');
		$foreign_key = static::foreign_key('folder');
		$key_for_relationship = static::key_for_relationship('folder');
		$values = array_merge($values, array($foreign_key => $this->$key_for_relationship));
		return Folder::create($values);
	}		
	
	function create_message($values){
		if(!$this->is_active) return $this->error->warning("I can't create a new message for ".$this->describe().' while it is inactive');
		if(!isset($this->id)) return $this->error->warning("I can't create a new message for ".$this->describe().' before it is saved to the database');
		
		$foreign_key = static::foreign_key('message');
		$values[$foreign_key] = $this->id;
		return Message::create($values);
	}

	function dn(){
		if($this->is_group){
			if(isset($this->id) && !$this->is_active) return get_instance()->groupsmodel->get_disabled_dn_from_groupname($this->name);
			return get_instance()->groupsmodel->get_dn_from_groupname($this->name);
		}
		if(isset($this->id) && !$this->is_active) return get_instance()->usersmodel->get_disabled_dn_from_username($this->name);
		return get_instance()->usersmodel->get_dn_from_username($this->name);
	}											 

	/** 
	* The email address for this mailbox.
	* @return string
	*/
	public function email_address(){
		if($this->property_is_empty('name')) return '';
		return $this->name.'@'.DIRECT_DOMAIN;
	}									 	

	function locations(){
		return array('inbox'=>'Inbox','sent'=>'Sent','draft'=>'Drafts','archived'=>'Archived') + collect('name', $mailbox->folders);
	}	
	
	/**
	* Statics on sent/received messagse for this mailbox & group mailboxes that the mailbox's owner belongs to.
	* This method only tracks sent/received messages - draft messages are not included.
	* @return array
	* @return array 
	* @return array
	*/
	function message_activity_statistics($conditions=array(), $group_mailboxes_to_include = array()){
		if(!is_array($conditions)) return $this->error->should_be_an_array_of_db_conditions($conditions);		
		if(!$this->is->array_of_nonempty_strings($group_mailboxes_to_include)) return $this->error->should_be_an_array_of_group_mailbox_names($group_mailboxes_to_include);
#TODO - HOW TO CHECK MEMBERSHIP FOR A USER?		
		if($this->is_group && !empty($group_mailboxes_to_include)) return $this->error->should_be_a_list_of_group_mailboxes_this_user_belongs_to($group_mailboxes_to_include);
			
		//find the mailboxes so that we can make sure that they're group mailboxes
		$group_mailboxes = array();
		if(!empty($group_mailboxes_to_include)){
			Mailbox::db()->where_in('name', $group_mailboxes_to_include);
			$group_mailboxes = Mailbox::find( array('is_group' => true));	
		}	
		$mailbox_ids = array_keys($group_mailboxes);
		array_unshift($mailbox_ids, $this->id);

		//find the counts for all of the mailboxes at once so that we can avoid excess db queries
		$counts = array();
		$conditions_for_counts = array( 'sent_total' => array_merge($conditions, array('sent' => true)),
										'sent_by_you' => array_merge($conditions, array('sent' => true, 'original_sender_id' => $this->id)),
										'received' => array_merge($conditions, array('sent' => false, 'draft' => false)),
										'total' => array_merge($conditions, array('draft' => false)) );
										
		foreach($conditions_for_counts as $count_type => $count_conditions){
			Message::db()->where_in('mailbox_id', $mailbox_ids);
			$counts[$count_type] = Message::count_by('mailbox_id', $count_conditions);		
		}										
		
		//separate them out by mailbox to make it easier to view
		$counts_by_mailbox = array();
		$i = 1;
		foreach($mailbox_ids as $mailbox_id){
			$mailbox = element($mailbox_id, $group_mailboxes, $this);
			$counts_by_mailbox[$mailbox->email_address] = array( 'default_order' => $i,
																 'address' => $mailbox->email_address,
																 'sent_by_you' => element($mailbox->id, $counts['sent_by_you'], 0),
																 'sent_total' => element($mailbox->id, $counts['sent_total'], 0),
																 'received' => element($mailbox->id, $counts['received'], 0),
																 'total' => element($mailbox->id, $counts['total'], 0),
																 );
																 
			$i++;
		}
			
		return $counts_by_mailbox;		
	}	
		
///////////////////////////////////////
// GETTERS
// For class vars prefixed with a _
///////////////////	///////////////////
	
	public function readonly_fields(){
		$readonly_fields = $this->_merge_with_parent_array('readonly_fields');
		//if the mailbox isn't active, all fields except for is_active are readonly
		if(isset($this->id) && !$this->is_active){
			$readonly_fields = array_unique( array_merge( $readonly_fields, array_diff( static::fields(), array('is_active') ) ) );
		}
		return $readonly_fields;
    }
	
	

	function __get($property){
		//for message relationships, allow developers to append the name of a message part to the relationship name
		//e.g. $this->inbox_message_headers will return the headers for $this->inbox_messages
		$suffix = substr($property, strrpos($property, '_') + 1);
		if(Message::is_a_part($suffix)){
			$relationship = strip_from_end('_'.$suffix, $property);
			if(static::has_relationship($relationship) && static::relationships($relationship, 'model') == 'message'){
				$this->set_up_relationship_query($relationship);
				return Message::find_part($suffix);
			}
		}
				
		return parent::__get($property);
	}
	
	function __call($name, $arguments){
		$class = get_called_class(); //doing this just to make phpdocumentor happier with it - can switch back to using static if we upgrade phpdocumentor
		
		//for message relationships, allow developers to append the name of a message part to the relationship name
		//e.g. $this->inbox_message_headers() will return the headers for $this->inbox_messages()
		$suffix = substr($name, strrpos($name, '_') + 1);
		if(Message::is_a_part($suffix)){
			$relationship = strip_from_end('_'.$suffix, $name);
			if($class::has_relationship($relationship) && $class::relationships($relationship, 'model') == 'message'){
				$this->set_up_relationship_query($relationship);
				return call_user_func_array('Message::find_part', array_merge( array($suffix), $arguments));
			}
		}
				
		return parent::__call($name, $arguments);
	}	
	
/////////////////////
// STATIC FUNCTIONS
/////////////////////
	
	public static function find_by_email_address($address){
		if(!get_instance()->is->string_like_a_direct_address($address)) return get_instance()->error->should_be_a_direct_address($address);
		$name = strip_from_end('@'.DIRECT_DOMAIN, $address);
		return static::find_one( compact('name') );
	}
	
/////////////////////////////////////////
// FUNCTIONS FOR edge application report
/////////////////////////////////////////

//these methods have been replaced by message_activity_statistics() & are being preserved for reference

/*
	function get_mailbox_id_by_name($name) {
		$mailbox = Mailbox::find_one(compact('name'));
		return $mailbox->id;
	}
	
	function get_sent_message_counts_grouped_by_addresses($username, $groupnames, $between) {
		$email_addresses = '';
		$user_email_address = $username . '@' .  DIRECT_DOMAIN;
		$mailbox_id = $this->get_mailbox_id_by_name($username);
		$groupname_array = explode(",", $groupnames);
		if(!empty($groupname_array)) {
			foreach($groupname_array as $gn) {
				if(!empty($gn)) {
					$email_addresses .= $this->db->escape($gn . '@' .  DIRECT_DOMAIN) . ',';
				}
			}
		}
		$email_addresses .= $this->db->escape($user_email_address);

		$sql_group = ' group by ms.sender ';
		$sql = ' select count(*) message_count, ms.sender sender ' . 
			   'from mail.dbo.mailboxes mb, mail.dbo.[messages] ms ' .
	 		   'where ms.sent = 1 ' .
 			   'and ms.mailbox_id = mb.id ' .
 			   'and ms.sender in (' . $email_addresses . ') ';
		
		if($between != null) {
			foreach ($between as $key => $value){//this if for anything that has an upper and lower boundary (ie date)
				if($value['end']){
					$sql .= "AND (ms.". $key ." <= ".$this->db->escape($value['end']).") ";
				}
				if($value['start']){
					$sql .= "AND (ms.". $key ." >= ".$this->db->escape($value['start']). ")";
				}
			}
		}
		//get all messages sent by the groups to which the user belongs
		$total_message_counts = $this->db->query($sql . $sql_group)->result();
		//get all messages sent by the user
		$total_message_counts_sent_by_user = $this->db->query($sql . ' and ms.original_sender_id = ' . $this->db->escape($mailbox_id) . $sql_group)->result();
		$message_counts = array();
		foreach($total_message_counts as $mc) {
			$sender = $mc->sender;
			//set the user's direct address to be first in the list
			if($sender == $user_email_address) { $message_counts[$sender]['default_order'] = 1; }
			else { $message_counts[$sender]['default_order'] = 2; }
			
			//have the same insert in both if and else to keep it in order for excel export
			$message_counts[$sender]['address'] = $sender;
			$message_counts[$sender]['sent_by_you'] = 0;
			$message_counts[$sender]['sent_total'] = $mc->message_count;
			//insert 0 to keep it in order for excel export
			$message_counts[$sender]['received'] = 0;
			$message_counts[$sender]['total'] = $mc->message_count;
		}
		foreach($total_message_counts_sent_by_user as $mc) {
			$sender = $mc->sender;
			$message_counts[$sender]['sent_by_you'] += $mc->message_count;
		}
		
		return $message_counts;
	}
	
	
	function get_received_message_counts_grouped_by_addresses($username, $groupnames, $between) {
		$groupname_array = explode(",", $groupnames);
		$user_email_address = $username . '@' .  DIRECT_DOMAIN;
		$sql = 'select count(*) message_count, ms.recipients recipients from ' .
  			   'mail.dbo.mailboxes mb, mail.dbo.[messages] ms ' .
  			   'where ms.sent = 1 and ms.mailbox_id = mb.id and (';
		if(!empty($groupname_array )) {
			foreach($groupname_array as $gn) {
				if(!empty($gn)) {
					$group_email_address = $gn . '@' .  DIRECT_DOMAIN;
					$sql .= 'ms.recipients like ' . $this->db->escape('%'.$group_email_address.'%') . ' or ';
				}
			}
		}
		$sql .= 'ms.recipients like ' . $this->db->escape('%'.$user_email_address.'%') . ') ';
	
		if($between != null) {
			foreach ($between as $key => $value){//this if for anything that has an upper and lower boundary (ie date)
				if($value['end']){
					$sql .= "AND (ms.". $key ." <= ".$this->db->escape($value['end']).") ";
				}
				if($value['start']){
					$sql .= "AND (ms.". $key ." >= ".$this->db->escape($value['start']). ")";
				}
			}
		}
		$sql .= 'group by ms.recipients';
		$received_message_counts = $this->db->query($sql)->result();
		$message_counts = array();
 		foreach($received_message_counts as $mc) {
 			$message_count = $mc->message_count;
			$recipients = json_decode($mc->recipients);
			if(!isset($recipients)) {
				$recipients = str_replace('[', '', $mc->recipients);
				$recipients = str_replace(']', '', $recipients);
			}
			//recipients column of messages table contains an array of recipients
			if(is_array($recipients)) {
				foreach($recipients as $recepient) {
					if(array_key_exists($recepient, $message_counts)) {
						$message_counts[$recepient] += $message_count;
					}
					else {
						$message_counts[$recepient] = $message_count;
					}
				}
			}
			//recipients column of messages table contains one recipient
			else {
				if(array_key_exists($recipients, $message_counts)) {
					$message_counts[$recipients] += $message_count;
				}
				else {
					$message_counts[$recipients] = $message_count;
				}
			}
 		}
		
		return $message_counts;
	}
	


	
	function get_message_counts_grouped_by_addresses($username, $groupnames, $between, $order, $direction, $is_export) {
		$sent_counts = $this->get_sent_message_counts_grouped_by_addresses($username, $groupnames, $between);
		$received_counts = $this->get_received_message_counts_grouped_by_addresses($username, $groupnames, $between);
		$name_array = explode(',', $groupnames);
		array_push($name_array, $username);
		$sent_by_you_from_all_addresses = 0;
		$sent_total_from_all_addresses = 0;
		$received_from_all_addresses = 0;
		$total_from_all_addresses = 0; 
		foreach($name_array as $name) {
			if(!empty($name)) {
				$address = $name.'@'.DIRECT_DOMAIN;
				if(array_key_exists($address, $received_counts)) {
					if(!array_key_exists($address, $sent_counts)) {
						if($name === $username) {
							$sent_counts[$address]['default_order'] = 1;
						}
						else {
							$sent_counts[$address]['default_order'] = 2;
						}
						$sent_counts[$address]['address'] = $address;
						$sent_counts[$address]['sent_by_you'] = 0;
						$sent_counts[$address]['sent_total'] = 0;
						$sent_counts[$address]['total'] = 0;
					}
					$sent_counts[$address]['received'] = $received_counts[$address];
					$sent_counts[$address]['total'] += $received_counts[$address];
				}
				
				if(!array_key_exists($address, $received_counts) && !array_key_exists($address, $sent_counts)) {
					if($name === $username) {
						$sent_counts[$address]['default_order'] = 1;
					}
					else {
						$sent_counts[$address]['default_order'] = 2;
					}
					$sent_counts[$address]['address'] = $address;
					$sent_counts[$address]['sent_by_you'] = 0;
					$sent_counts[$address]['sent_total'] = 0;
					$sent_counts[$address]['received'] = 0;
					$sent_counts[$address]['total'] = 0;
				}
				if(array_key_exists($address, $sent_counts)) {
					$sent_by_you_from_all_addresses += $sent_counts[$address]['sent_by_you'];
					$sent_total_from_all_addresses += $sent_counts[$address]['sent_total'];
					$received_from_all_addresses += $sent_counts[$address]['received'];
					$total_from_all_addresses += $sent_counts[$address]['total'];
				}
			}
		}
		$this->_mailbox_nested_array_sort($sent_counts, $order, $direction);
		
		$sent_counts['Total VA:']['default_order'] = 3;
		$sent_counts['Total VA:']['address'] = 'Total VA:';
		$sent_counts['Total VA:']['sent_by_you'] = $sent_by_you_from_all_addresses;
		$sent_counts['Total VA:']['sent_total'] = $sent_total_from_all_addresses;
		$sent_counts['Total VA:']['received'] = $received_from_all_addresses;
		$sent_counts['Total VA:']['total'] = $total_from_all_addresses;
		if($is_export == 'excel') {
			foreach($sent_counts as $addrs => $sent_count) {
				unset($sent_count['default_order']);
				$sent_counts[$addrs] = $sent_count;
			} 
		}
		return $sent_counts;
	}
	
	private function _mailbox_nested_array_sort(&$array, $key, $direction) {
		if(strtoupper($key) == 'ADDRESS') {
			if(strtoupper($direction) == 'ASC') {
				ksort($array);
			}
			else {
				krsort($array);
			}
		}
		else {
			$sorter = array();
			$ret = array();
			reset($array);
			foreach($array as $ii => $va) {
				$sorter[$ii] = $va[$key];
			}
			$domains = array();
			$msg_count = array();
			foreach($sorter as $key => $value) {
				$domains[] = $key;
				$msg_count[] = $value;
			}
			if(strtoupper($direction) == 'ASC') {
				array_multisort($msg_count, SORT_ASC, $domains, SORT_ASC, $sorter);
			}
			else {
				array_multisort($msg_count, SORT_DESC, $domains, SORT_ASC, $sorter);
			}
			foreach($sorter as $ii => $va) {
				$ret[$ii] = $array[$ii];
			}
			$array = $ret;
		}
	}
*/	
	
}
